// src/app/controllers/api/app/threads.controller.ts
import type { Request, Response } from "express";
import { z } from "zod";
import { withRlsTx, type Db } from "@/lib/postgres.js";
import { requireCaseRole } from "../../../../utils/aclCase.js";
import { writeAuditLog } from "../../../../audit/writeAuditLog.js";
const UUID_RE =
  /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;

async function resolveCaseId(db: any, caseRef: string): Promise<string> {
  if (UUID_RE.test(caseRef)) return z.string().uuid().parse(caseRef);
  const r = await db.query(
    `SELECT id FROM public.cases WHERE code=$1 LIMIT 1`,
    [caseRef.trim()]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Case not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0].id as string;
}

async function resolveThreadId(db: any, threadRef: string): Promise<string> {
  if (UUID_RE.test(threadRef)) return z.string().uuid().parse(threadRef);
  const r = await db.query(
    `SELECT id FROM public.case_threads WHERE code=$1 LIMIT 1`,
    [threadRef.trim()]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Thread not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0].id as string;
}
function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}

export async function listThreadsByCase(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const includeArchived =
      req.query.includeArchived === "1" || req.query.includeArchived === "true";

    const threads = await withRlsTx(rlsCtx(req), async (db) => {
      const caseId = await resolveCaseId(db, caseRef);

      const { rows } = await db.query(
        `SELECT
           id, code, case_id, title, description, input_due_at,
           created_by, created_at, updated_at,
           archived_at, archived_by
         FROM public.case_threads
         WHERE case_id=$1
           AND ($2::boolean = true OR archived_at IS NULL)
         ORDER BY created_at DESC`,
        [caseId, includeArchived]
      );
      return rows;
    });

    res.json({ threads });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listThreadsByCase", message: e.message });
  }
}

const CreateThreadBody = z.object({
  title: z.string().min(2),
  description: z.string().optional(),
  inputDueAt: z.string().datetime().optional(),
});

export async function createThread(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const b = CreateThreadBody.parse(req.body);

    const created = await withRlsTx(rlsCtx(req), async (db) => {
      const caseId = await resolveCaseId(db, caseRef);

      await requireCaseRole(db, caseId, "editor");

      const { rows } = await db.query<{ id: string; code: string }>(
        `
        INSERT INTO public.case_threads (id, case_id, title, description, input_due_at, created_by)
        VALUES (gen_random_uuid(), $1, $2, $3, $4, $5)
        RETURNING id, code
        `,
        [
          caseId,
          b.title.trim(),
          b.description ?? null,
          b.inputDueAt ? new Date(b.inputDueAt) : null,
          req.user!.id,
        ]
      );
      return rows[0];
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "THREAD_CREATED",
      objectType: "thread",
      objectId: created.id,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { case_ref: caseRef, title: b.title, code: created.code },
    });

    res.status(201).json({ id: created.id, code: created.code });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "createThread", message: e.message });
  }
}

export async function listThreadComments(req: Request, res: Response) {
  try {
    const threadRef = z.string().min(1).parse(req.params.threadRef);

    const comments = await withRlsTx(rlsCtx(req), async (db) => {
      const threadId = await resolveThreadId(db, threadRef);

      const { rows } = await db.query(
        `SELECT
  c.id, c.thread_id, c.author_id,
  u.name as author_name,
  c.comment_type, c.body, c.created_at, c.updated_at
FROM public.thread_comments c
LEFT JOIN public.users u ON u.id = c.author_id
WHERE c.thread_id=$1
ORDER BY c.created_at ASC`,
        [threadId]
      );
      return rows;
    });

    res.json({ comments });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listThreadComments", message: e.message });
  }
}

const CreateCommentBody = z.object({
  body: z.string().min(1),
  commentType: z
    .enum(["comment", "question", "proposal", "stance"])
    .default("comment"),
});

export async function createThreadComment(req: Request, res: Response) {
  try {
    const threadRef = z.string().min(1).parse(req.params.threadRef);
    const b = CreateCommentBody.parse(req.body);

    const created = await withRlsTx(rlsCtx(req), async (db) => {
      const threadId = await resolveThreadId(db, threadRef);

      const { rows: ins } = await db.query<{ id: string }>(
        `
        INSERT INTO public.thread_comments (id, thread_id, author_id, comment_type, body)
        VALUES (gen_random_uuid(), $1, $2, $3, $4)
        RETURNING id
        `,
        [threadId, req.user!.id, b.commentType, b.body]
      );
      return ins[0].id;
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "THREAD_COMMENT_CREATED",
      objectType: "thread_comment",
      objectId: created,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { thread_ref: threadRef, type: b.commentType },
    });

    res.status(201).json({ id: created });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "createThreadComment", message: e.message });
  }
}
export async function listThreadAttachments(req: Request, res: Response) {
  try {
    const threadRef = z.string().min(1).parse(req.params.threadRef);

    const attachments = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const threadId = await resolveThreadId(db, threadRef);

      const { rows } = await db.query<{
        id: string;
        original_name: string;
        mime_type: string | null;
        size_bytes: string | number | null;
        uploaded_at: string;
        uploaded_by: string | null;
      }>(
        `
        SELECT
          a.id,
          a.original_name,
          a.mime_type,
          a.size_bytes,
          a.uploaded_at,
          a.uploaded_by
        FROM public.thread_attachments ta
        JOIN public.attachments a ON a.id = ta.attachment_id
        JOIN public.case_threads t ON t.id = ta.thread_id
        JOIN public.case_members m ON m.case_id = t.case_id
        WHERE ta.thread_id = $1
          AND m.user_id = public.app_user_id()
          AND m.is_active = true
        ORDER BY a.uploaded_at DESC
        `,
        [threadId]
      );

      return rows.map((r) => ({
        ...r,
        downloadUrl: `/api/app/attachments/${r.id}/download`,
      }));
    });

    res.json({ attachments });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listThreadAttachments", message: e.message });
  }
}

async function getThreadCaseAndAuthor(db: Db, threadId: string) {
  const r = await db.query(
    `SELECT id, case_id, created_by, archived_at
     FROM public.case_threads
     WHERE id=$1`,
    [threadId]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Thread not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0] as {
    id: string;
    case_id: string;
    created_by: string | null;
    archived_at: string | null;
  };
}

async function canEditByAuthorOrEditor(
  db: Db,
  caseId: string,
  createdBy: string | null,
  userId: string
) {
  if (createdBy && createdBy === userId) return true;
  // ha nem szerző, akkor editor kell
  await requireCaseRole(db, caseId, "editor");
  return true;
}

const PatchThreadBody = z.object({
  title: z.string().min(2).optional(),
  // ⚠️ nullable: lehessen törölni
  description: z.string().nullable().optional(),
  // ⚠️ nullable: inputDueAt törléshez
  inputDueAt: z.string().datetime().nullable().optional(),
});

export async function patchThread(req: Request, res: Response) {
  try {
    const threadRef = z.string().min(1).parse(req.params.threadRef);
    const b = PatchThreadBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const threadId = await resolveThreadId(db, threadRef);

      const t = await getThreadCaseAndAuthor(db, threadId);
      await canEditByAuthorOrEditor(db, t.case_id, t.created_by, req.user!.id);

      const hasTitle = b.title !== undefined;
      const hasDescription = b.description !== undefined;
      const hasInputDueAt = b.inputDueAt !== undefined;

      await db.query(
        `
  UPDATE public.case_threads
  SET
    title = CASE
      WHEN $2::boolean THEN $3
      ELSE title
    END,
    description = CASE
      WHEN $4::boolean THEN $5
      ELSE description
    END,
    input_due_at = CASE
      WHEN $6::boolean THEN $7
      ELSE input_due_at
    END,
    updated_by = $8,
    updated_at = now()
  WHERE id = $1
  `,
        [
          threadId,

          // title
          hasTitle,
          hasTitle ? b.title!.trim() : null,

          // description (nullable)
          hasDescription,
          hasDescription ? b.description ?? null : null,

          // inputDueAt (nullable)
          hasInputDueAt,
          hasInputDueAt ? (b.inputDueAt ? new Date(b.inputDueAt) : null) : null,

          req.user!.id,
        ]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "THREAD_UPDATED",
      objectType: "thread",
      objectId: threadRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { thread_ref: threadRef, changed: Object.keys(b) },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "patchThread", message: e.message });
  }
}

export async function archiveThread(req: Request, res: Response) {
  try {
    const threadRef = z.string().min(1).parse(req.params.threadRef);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const threadId = await resolveThreadId(db, threadRef);

      const t = await getThreadCaseAndAuthor(db, threadId);
      await canEditByAuthorOrEditor(db, t.case_id, t.created_by, req.user!.id);

      await db.query(
        `
        UPDATE public.case_threads
        SET archived_at=now(), archived_by=$2, updated_by=$2, updated_at=now()
        WHERE id=$1 AND archived_at IS NULL
        `,
        [threadId, req.user!.id]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "THREAD_ARCHIVED",
      objectType: "thread",
      objectId: threadRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { thread_ref: threadRef },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "archiveThread", message: e.message });
  }
}

// ---------- COMMENTS ----------

async function getCommentWithThread(db: Db, commentId: string) {
  const r = await db.query(
    `
    SELECT
      c.id,
      c.thread_id,
      c.author_id,
      c.archived_at,
      t.case_id
    FROM public.thread_comments c
    JOIN public.case_threads t ON t.id = c.thread_id
    WHERE c.id=$1
    `,
    [commentId]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Comment not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0] as {
    id: string;
    thread_id: string;
    author_id: string | null;
    archived_at: string | null;
    case_id: string;
  };
}

const PatchCommentBody = z.object({
  body: z.string().min(1).optional(),
  commentType: z.enum(["comment", "question", "proposal", "stance"]).optional(),
});

export async function patchThreadComment(req: Request, res: Response) {
  try {
    const threadRef = z.string().min(1).parse(req.params.threadRef);
    const commentId = z.string().uuid().parse(req.params.commentId);
    const b = PatchCommentBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const threadIdFromUrl = await resolveThreadId(db, threadRef);

      const c = await getCommentWithThread(db, commentId);

      // ⚠️ Ha a comment nem ehhez a threadhez tartozik → 404
      if (c.thread_id !== threadIdFromUrl) {
        const e: any = new Error("Comment not found");
        e.statusCode = 404;
        throw e;
      }

      await canEditByAuthorOrEditor(db, c.case_id, c.author_id, req.user!.id);

      await db.query(
        `
        UPDATE public.thread_comments
        SET
          body        = COALESCE($2, body),
          comment_type= COALESCE($3, comment_type),
          updated_by  = $4,
          updated_at  = now()
        WHERE id=$1 AND archived_at IS NULL
        `,
        [commentId, b.body ?? null, b.commentType ?? null, req.user!.id]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "THREAD_COMMENT_UPDATED",
      objectType: "thread_comment",
      objectId: commentId,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { changed: Object.keys(b) },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "patchThreadComment", message: e.message });
  }
}

export async function archiveThreadComment(req: Request, res: Response) {
  try {
    const threadRef = z.string().min(1).parse(req.params.threadRef);
    const commentId = z.string().uuid().parse(req.params.commentId);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const threadIdFromUrl = await resolveThreadId(db, threadRef);

      const c = await getCommentWithThread(db, commentId);
      if (c.thread_id !== threadIdFromUrl) {
        const e: any = new Error("Comment not found");
        e.statusCode = 404;
        throw e;
      }

      await canEditByAuthorOrEditor(db, c.case_id, c.author_id, req.user!.id);

      await db.query(
        `
        UPDATE public.thread_comments
        SET archived_at=now(), archived_by=$2, updated_by=$2, updated_at=now()
        WHERE id=$1 AND archived_at IS NULL
        `,
        [commentId, req.user!.id]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "THREAD_COMMENT_ARCHIVED",
      objectType: "thread_comment",
      objectId: commentId,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: {},
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "archiveThreadComment", message: e.message });
  }
}
export async function getThreadDetail(req: Request, res: Response) {
  try {
    const threadRef = z.string().min(1).parse(req.params.threadRef);

    const thread = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const threadId = await resolveThreadId(db, threadRef);

      const { rows } = await db.query(
        `SELECT
           id, code, case_id, title, description, input_due_at,
           created_by, created_at, updated_at,
           archived_at, archived_by
         FROM public.case_threads
         WHERE id=$1
         LIMIT 1`,
        [threadId]
      );

      if (!rows.length) return null;
      return rows[0];
    });

    if (!thread) return res.status(404).json({ error: "Not found" });
    res.json({ thread });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "getThreadDetail", message: e.message });
  }
}
