import type { Request, Response } from "express";
import { z } from "zod";
import { withRlsTx, type Db } from "@/lib/postgres.js";

function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}

const UUID_RE =
  /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;

async function resolvePlaybookId(db: Db, playbookRef: string): Promise<string> {
  const ref = String(playbookRef ?? "").trim();
  if (!ref) {
    const e: any = new Error("Playbook ref missing");
    e.statusCode = 400;
    throw e;
  }

  if (UUID_RE.test(ref)) return z.string().uuid().parse(ref);

  const r = await db.query<{ id: string }>(
    `SELECT id FROM public.playbooks WHERE key=$1 LIMIT 1`,
    [ref]
  );

  if (!r.rows?.length) {
    const e: any = new Error("Playbook not found");
    e.statusCode = 404;
    throw e;
  }

  return r.rows[0].id;
}

const BoolQuery = z
  .union([
    z.literal("1"),
    z.literal("0"),
    z.literal("true"),
    z.literal("false"),
  ])
  .optional()
  .transform((v) => v === "1" || v === "true");

const ListPlaybooksQuery = z.object({
  includeInactive: BoolQuery, // boolean lesz
});

export async function listPlaybooks(req: Request, res: Response) {
  try {
    const q = ListPlaybooksQuery.parse(req.query);

    const rows = await withRlsTx(rlsCtx(req), async (db) => {
      const includeInactive = !!q.includeInactive;

      const r = await db.query(
        `
        SELECT
          id, key, name, description,
          default_case_type, default_flow_type,
          is_active,
          created_at, updated_at
        FROM public.playbooks
        WHERE ($1::boolean = true OR is_active = true)
        ORDER BY is_active DESC, name ASC
        `,
        [includeInactive]
      );

      return r.rows;
    });

    res.json({ playbooks: rows });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listPlaybooks", message: e.message });
  }
}

export async function getPlaybookDetail(req: Request, res: Response) {
  try {
    const playbookRef = z.string().min(1).parse(req.params.playbookRef);

    const out = await withRlsTx(rlsCtx(req), async (db) => {
      const playbookId = await resolvePlaybookId(db, playbookRef);

      const { rows: pb } = await db.query(
        `
        SELECT
          id, key, name, description,
          default_case_type, default_flow_type,
          is_active,
          created_at, updated_at
        FROM public.playbooks
        WHERE id=$1
        LIMIT 1
        `,
        [playbookId]
      );

      if (!pb.length) return null;

      const { rows: steps } = await db.query(
        `
        SELECT
          id, playbook_id, sort_order, title, description
        FROM public.playbook_steps
        WHERE playbook_id=$1
        ORDER BY sort_order ASC, title ASC
        `,
        [playbookId]
      );

      // A-opció: a description mezőben lehet “Cadence:” és “Roles:” sor (konvenció)
      // FE-n ezt szépen ki lehet szedni és megjeleníteni, de nem kötelező.
      return { playbook: pb[0], steps };
    });

    if (!out) return res.status(404).json({ error: "Not found" });
    res.json(out);
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "getPlaybookDetail", message: e.message });
  }
}
