// src/app/controllers/api/admin/playbooks/playbooks.admin.controller.ts
import type { Request, Response } from "express";
import { z } from "zod";
import { withRlsTx, type Db } from "@/lib/postgres.js";

function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}

function requireSuperAdmin(req: Request) {
  if (req.user?.accessLevel?.name !== "super-admin") {
    const e: any = new Error("Forbidden");
    e.statusCode = 403;
    throw e;
  }
}

const UUID_RE =
  /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;

async function resolvePlaybookId(db: Db, playbookRef: string): Promise<string> {
  const ref = String(playbookRef ?? "").trim();
  if (!ref) {
    const e: any = new Error("Playbook ref missing");
    e.statusCode = 400;
    throw e;
  }
  if (UUID_RE.test(ref)) return z.string().uuid().parse(ref);

  const r = await db.query<{ id: string }>(
    `SELECT id FROM public.playbooks WHERE key=$1 LIMIT 1`,
    [ref]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Playbook not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0].id;
}

const CreatePlaybookBody = z.object({
  key: z.string().min(2),
  name: z.string().min(2),
  description: z.string().optional(),
  defaultCaseType: z.string().nullable().optional(),
  defaultFlowType: z.enum(["transient", "continuous"]).nullable().optional(),
  isActive: z.boolean().optional(),
});

export async function createPlaybook(req: Request, res: Response) {
  try {
    requireSuperAdmin(req);
    const b = CreatePlaybookBody.parse(req.body);

    const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const r = await db.query(
        `
        INSERT INTO public.playbooks (
          id, key, name, description,
          default_case_type, default_flow_type,
          is_active, created_by, updated_by
        )
        VALUES (
          gen_random_uuid(),
          $1, $2, $3,
          $4, $5,
          COALESCE($6, true),
          $7, $7
        )
        RETURNING
          id, key, name, description,
          default_case_type, default_flow_type,
          is_active, created_at, updated_at
        `,
        [
          b.key.trim(),
          b.name.trim(),
          b.description ?? null,
          b.defaultCaseType ?? null,
          b.defaultFlowType ?? null,
          b.isActive ?? true,
          req.user!.id,
        ]
      );
      return r.rows[0];
    });

    res.status(201).json({ playbook: out });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "createPlaybook", message: e.message });
  }
}

const UpdatePlaybookBody = z.object({
  name: z.string().min(2).optional(),
  description: z.string().nullable().optional(),
  defaultCaseType: z.string().nullable().optional(),
  defaultFlowType: z.enum(["transient", "continuous"]).nullable().optional(),
  isActive: z.boolean().optional(),
});

export async function updatePlaybook(req: Request, res: Response) {
  try {
    requireSuperAdmin(req);
    const playbookRef = z.string().min(1).parse(req.params.playbookRef);
    const b = UpdatePlaybookBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const id = await resolvePlaybookId(db, playbookRef);

      const sets: string[] = [];
      const params: any[] = [id];

      const push = (sql: string, val: any) => {
        params.push(val);
        sets.push(sql.replace("$X", `$${params.length}`));
      };

      if (b.name !== undefined) push(`name=$X`, b.name.trim());
      if (b.description !== undefined)
        push(`description=$X`, b.description ?? null);
      if (b.defaultCaseType !== undefined)
        push(`default_case_type=$X`, b.defaultCaseType ?? null);
      if (b.defaultFlowType !== undefined)
        push(`default_flow_type=$X`, b.defaultFlowType ?? null);
      if (b.isActive !== undefined) push(`is_active=$X`, b.isActive);

      // always bump updated_by/updated_at if anything changed
      if (!sets.length) return;

      params.push(req.user!.id);
      await db.query(
        `
        UPDATE public.playbooks
        SET ${sets.join(", ")},
            updated_by=$${params.length},
            updated_at=now()
        WHERE id=$1
        `,
        params
      );
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "updatePlaybook", message: e.message });
  }
}

const CreateStepBody = z.object({
  sortOrder: z.number().int().min(0).default(0),
  title: z.string().min(2),
  description: z.string().nullable().optional(),
});

export async function createPlaybookStep(req: Request, res: Response) {
  try {
    requireSuperAdmin(req);
    const playbookRef = z.string().min(1).parse(req.params.playbookRef);
    const b = CreateStepBody.parse(req.body);

    const out = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const playbookId = await resolvePlaybookId(db, playbookRef);

      const r = await db.query(
        `
        INSERT INTO public.playbook_steps (id, playbook_id, sort_order, title, description)
        VALUES (gen_random_uuid(), $1, $2, $3, $4)
        RETURNING id, playbook_id, sort_order, title, description
        `,
        [playbookId, b.sortOrder, b.title.trim(), b.description ?? null]
      );
      return r.rows[0];
    });

    res.status(201).json({ step: out });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "createPlaybookStep", message: e.message });
  }
}

const UpdateStepBody = z.object({
  sortOrder: z.number().int().min(0).optional(),
  title: z.string().min(2).optional(),
  description: z.string().nullable().optional(),
});

export async function updatePlaybookStep(req: Request, res: Response) {
  try {
    requireSuperAdmin(req);

    const stepId = z.string().uuid().parse(req.params.stepId);
    const b = UpdateStepBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const sets: string[] = [];
      const params: any[] = [stepId];

      const push = (sql: string, val: any) => {
        params.push(val);
        sets.push(sql.replace("$X", `$${params.length}`));
      };

      if (b.sortOrder !== undefined) push(`sort_order=$X`, b.sortOrder);
      if (b.title !== undefined) push(`title=$X`, b.title.trim());
      if (b.description !== undefined)
        push(`description=$X`, b.description ?? null);

      if (!sets.length) return;

      await db.query(
        `UPDATE public.playbook_steps SET ${sets.join(", ")} WHERE id=$1`,
        params
      );
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "updatePlaybookStep", message: e.message });
  }
}

export async function deletePlaybookStep(req: Request, res: Response) {
  try {
    requireSuperAdmin(req);
    const stepId = z.string().uuid().parse(req.params.stepId);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      await db.query(`DELETE FROM public.playbook_steps WHERE id=$1`, [stepId]);
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "deletePlaybookStep", message: e.message });
  }
}
