// src/app/controllers/api/admin/meetings/meetings.action-items.controller.ts
import type { Request, Response } from "express";
import { z } from "zod";
import { withRlsTx, type Db } from "@/lib/postgres.js";
import { requireCaseRole } from "../../../../utils/aclCase.js";
import { writeAuditLog } from "../../../../audit/writeAuditLog.js";

function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}
async function resolveMeetingId(db: Db, meetingRef: string): Promise<string> {
  // uuid → id
  if (/^[0-9a-fA-F-]{36}$/.test(meetingRef)) return meetingRef;

  const { rows } = await db.query<{ id: string }>(
    `SELECT id FROM public.meetings WHERE code = $1 LIMIT 1`,
    [meetingRef]
  );

  if (!rows.length) {
    const e: any = new Error("Meeting not found");
    e.statusCode = 404;
    throw e;
  }
  return rows[0].id;
}

const CreateFromAgendaBody = z.object({
  agendaItemId: z.string().uuid(),
  title: z.string().min(2).optional(), // ha nincs, az agenda item title-ját vesszük
  description: z.string().optional(),
  assignedTo: z.string().uuid().optional(),
  dueAt: z.string().datetime().optional(),
});

export async function createActionItemFromAgendaItem(
  req: Request,
  res: Response
) {
  try {
    const meetingRef = z.string().min(1).parse(req.params.meetingRef);
    const b = CreateFromAgendaBody.parse(req.body);

    const created = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const meetingId = await resolveMeetingId(db, meetingRef);
      // 1) agenda item betöltése (meeting + case + thread)
      const { rows: ai } = await db.query<{
        id: string;
        meeting_id: string;
        case_id: string;
        thread_id: string | null;
        title: string;
      }>(
        `
        SELECT id, meeting_id, case_id, thread_id, title
        FROM public.meeting_agenda_items
        WHERE id = $1 AND meeting_id = $2
        LIMIT 1
        `,
        [b.agendaItemId, meetingId]
      );

      if (!ai.length) {
        const e: any = new Error("Agenda item not found");
        e.statusCode = 404;
        throw e;
      }

      // 2) jogosultság: case editor (vagy super-admin RLS)
      await requireCaseRole(db, ai[0].case_id, "editor");

      // 3) action item létrehozás
      const title = (
        b.title?.trim() ||
        ai[0].title?.trim() ||
        "Action item"
      ).slice(0, 500);

      const { rows: createdRows } = await db.query<{ id: string }>(
        `
        INSERT INTO public.action_items (
          id, title, description, status,
          case_id, thread_id, meeting_id,
          assigned_to, created_by, due_at
        )
        VALUES (
          gen_random_uuid(), $1, $2, 'open'::public.action_item_status,
          $3, $4, $5,
          $6, $7, $8
        )
        RETURNING id
        `,
        [
          title,
          b.description ?? null,
          ai[0].case_id,
          ai[0].thread_id ?? null,
          meetingId,
          b.assignedTo ?? null,
          req.user!.id,
          b.dueAt ? new Date(b.dueAt) : null,
        ]
      );

      return {
        actionItemId: createdRows[0].id,
        caseId: ai[0].case_id,
        threadId: ai[0].thread_id,
        meetingId,
      };
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "ACTION_ITEM_CREATED_FROM_AGENDA",
      objectType: "meeting",
      objectId: created.meetingId,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: {
        meeting_ref: meetingRef,
        meeting_id: created.meetingId,
        agenda_item_id: b.agendaItemId,
        action_item_id: created.actionItemId,
        case_id: created.caseId,
        thread_id: created.threadId,
        assigned_to: b.assignedTo ?? null,
        due_at: b.dueAt ?? null,
      },
    });

    res.status(201).json({ id: created.actionItemId });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "createActionItemFromAgendaItem", message: e.message });
  }
}
