// src/app/controllers/api/app/decisions.controller.ts
import type { Request, Response } from "express";
import { z } from "zod";
import { withRlsTx, type Db } from "@/lib/postgres.js";
import { requireCaseRole } from "../../../../utils/aclCase.js";
import { writeAuditLog } from "../../../../audit/writeAuditLog.js";
const UUID_RE =
  /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;

async function resolveCaseId(db: any, caseRef: string): Promise<string> {
  if (UUID_RE.test(caseRef)) return z.string().uuid().parse(caseRef);
  const r = await db.query(
    `SELECT id FROM public.cases WHERE code=$1 LIMIT 1`,
    [caseRef.trim()]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Case not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0].id as string;
}
async function resolveUserId(
  db: any,
  key: { id?: string; email?: string; username?: string }
): Promise<string> {
  if (key.id) return z.string().uuid().parse(key.id);

  if (key.email) {
    const r = await db.query(
      `SELECT id FROM public.users WHERE lower(email)=lower($1) LIMIT 1`,
      [key.email.trim()]
    );
    const rows = (r.rows ?? []) as Array<{ id: string }>;
    if (!rows.length)
      throw new Error(`User nem található email alapján: ${key.email}`);
    return rows[0].id;
  }

  if (key.username) {
    const r = await db.query(
      `SELECT id FROM public.users WHERE lower(username)=lower($1) LIMIT 1`,
      [key.username.trim()]
    );
    const rows = (r.rows ?? []) as Array<{ id: string }>;
    if (!rows.length)
      throw new Error(`User nem található username alapján: ${key.username}`);
    return rows[0].id;
  }

  throw new Error("User azonosító hiányzik (id/email/username).");
}
async function resolveDecisionId(
  db: any,
  decisionRef: string
): Promise<string> {
  if (UUID_RE.test(decisionRef)) return z.string().uuid().parse(decisionRef);
  const r = await db.query(
    `SELECT id FROM public.decisions WHERE code=$1 LIMIT 1`,
    [decisionRef.trim()]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Decision not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0].id as string;
}
function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}

export async function listDecisionsByCase(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const includeArchived =
      req.query.includeArchived === "1" || req.query.includeArchived === "true";

    const decisions = await withRlsTx(rlsCtx(req), async (db) => {
      const caseId = await resolveCaseId(db, caseRef);

      const { rows } = await db.query(
        `SELECT
           id, code, case_id, thread_id, meeting_id,
           title, decision_text, status, decided_at, owner_user_id, due_at,
           created_at, updated_at, archived_at, archived_by
         FROM public.decisions
         WHERE case_id=$1
           AND ($2::boolean = true OR archived_at IS NULL)
         ORDER BY created_at DESC`,
        [caseId, includeArchived]
      );
      return rows;
    });

    res.json({ decisions });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listDecisionsByCase", message: e.message });
  }
}

const CreateDecisionBody = z.object({
  title: z.string().min(2),
  decisionText: z.string().min(2),
  threadId: z.string().uuid().optional(),
  meetingId: z.string().uuid().optional(),

  // legacy
  ownerUserId: z.string().uuid().optional(),
  // ✅ human readable: email | username | uuid
  ownerUser: z.string().min(1).optional(),

  dueAt: z.string().datetime().optional(),
});

export async function createDecision(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const b = CreateDecisionBody.parse(req.body);

    const created = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const caseId = await resolveCaseId(db, caseRef);

      await requireCaseRole(db, caseId, "editor");

      // ✅ owner feloldás (uuid/email/username)
      let ownerId: string | null = null;

      if (b.ownerUserId) {
        ownerId = b.ownerUserId;
      } else if (b.ownerUser) {
        const raw = b.ownerUser.trim();
        const asUuid = z.string().uuid().safeParse(raw);

        if (asUuid.success) ownerId = asUuid.data;
        else if (raw.includes("@"))
          ownerId = await resolveUserId(db, { email: raw });
        else ownerId = await resolveUserId(db, { username: raw });
      }

      const { rows } = await db.query<{ id: string; code: string }>(
        `
        INSERT INTO public.decisions
          (id, case_id, thread_id, meeting_id, title, decision_text, status, owner_user_id, due_at, created_by)
        VALUES
          (gen_random_uuid(), $1, $2, $3, $4, $5, 'draft'::public.decision_status, $6, $7, $8)
        RETURNING id, code
        `,
        [
          caseId,
          b.threadId ?? null,
          b.meetingId ?? null,
          b.title.trim(),
          b.decisionText,
          ownerId, // ✅ EZ volt a bug: ne b.ownerUserId-t küldd
          b.dueAt ? new Date(b.dueAt) : null,
          req.user!.id,
        ]
      );

      return rows[0];
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "DECISION_CREATED",
      objectType: "decision",
      objectId: created.id,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { case_ref: caseRef, code: created.code },
    });

    res.status(201).json({ id: created.id, code: created.code });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "createDecision", message: e.message });
  }
}

const UpdateDecisionStatusBody = z.object({
  status: z.enum(["draft", "approved", "rejected", "deferred", "cancelled"]),
  decidedAt: z.string().datetime().optional(), // if setting approved/rejected/deferred
});
export async function updateDecisionStatus(req: Request, res: Response) {
  try {
    const decisionRef = z.string().min(1).parse(req.params.decisionRef);
    const b = UpdateDecisionStatusBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db) => {
      const decisionId = await resolveDecisionId(db, decisionRef);

      // We need the case_id to check role
      const { rows } = await db.query<{ case_id: string }>(
        `SELECT case_id FROM public.decisions WHERE id=$1`,
        [decisionId]
      );
      if (!rows.length) {
        const e: any = new Error("Decision not found");
        e.statusCode = 404;
        throw e;
      }
      await requireCaseRole(db, rows[0].case_id, "editor");

      await db.query(
        `
        UPDATE public.decisions
        SET status=$2::public.decision_status,
            decided_at=COALESCE($3, decided_at),
            updated_at=now()
        WHERE id=$1
        `,
        [
          decisionId,
          b.status,
          b.decidedAt
            ? new Date(b.decidedAt)
            : ["approved", "rejected", "deferred"].includes(b.status)
            ? new Date()
            : null,
        ]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "DECISION_STATUS_UPDATED",
      objectType: "decision",
      objectId: decisionRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { decision_ref: decisionRef, status: b.status },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "updateDecisionStatus", message: e.message });
  }
}

// --- Globális Decision Log ---
const ListDecisionsQuery = z.object({
  status: z
    .enum(["draft", "approved", "rejected", "deferred", "cancelled"])
    .optional(),
  from: z.string().datetime().optional(),
  to: z.string().datetime().optional(),
  q: z.string().optional(),

  // régi
  caseId: z.string().uuid().optional(),

  // ✅ új
  caseRef: z.string().min(1).optional(),

  meetingId: z.string().uuid().optional(),

  includeArchived: z.union([z.literal("1"), z.literal("true")]).optional(),
});

export async function listDecisions(req: Request, res: Response) {
  try {
    const q = ListDecisionsQuery.parse(req.query);

    const decisions = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const params: any[] = [];
      const where: string[] = [];

      if (!q.includeArchived) where.push(`d.archived_at IS NULL`);

      if (q.status) {
        params.push(q.status);
        where.push(`d.status = $${params.length}::public.decision_status`);
      }
      if (q.from) {
        params.push(new Date(q.from));
        where.push(`d.created_at >= $${params.length}`);
      }
      if (q.to) {
        params.push(new Date(q.to));
        where.push(`d.created_at < $${params.length}`);
      }

      if (q.caseId) {
        params.push(q.caseId);
        where.push(`d.case_id = $${params.length}::uuid`);
      } else if (q.caseRef) {
        const caseId = await resolveCaseId(db, q.caseRef);
        params.push(caseId);
        where.push(`d.case_id = $${params.length}::uuid`);
      }

      if (q.meetingId) {
        params.push(q.meetingId);
        where.push(`d.meeting_id = $${params.length}::uuid`);
      }

      if (q.q?.trim()) {
        params.push(`%${q.q.trim()}%`);
        where.push(
          `(d.code ILIKE $${params.length} OR d.title ILIKE $${params.length} OR d.decision_text ILIKE $${params.length})`
        );
      }

      const sql = `
        SELECT
          d.id, d.code, d.case_id, d.thread_id, d.meeting_id,
          d.title, d.decision_text, d.status, d.decided_at,
          d.owner_user_id, d.due_at, d.created_at, d.updated_at,
          d.archived_at, d.archived_by
        FROM public.decisions d
        ${where.length ? `WHERE ${where.join(" AND ")}` : ""}
        ORDER BY d.updated_at DESC
        LIMIT 200
      `;
      const { rows } = await db.query(sql, params);
      return rows;
    });

    res.json({ decisions });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listDecisions", message: e.message });
  }
}

export async function getDecisionDetail(req: Request, res: Response) {
  try {
    const decisionRef = z.string().min(1).parse(req.params.decisionRef);

    const data = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const decisionId = await resolveDecisionId(db, decisionRef);

      const { rows: d } = await db.query(
        `
        SELECT
          id, code, case_id, thread_id, meeting_id, title, decision_text, status,
          decided_at, owner_user_id, due_at, created_at, updated_at,
          archived_at, archived_by
        FROM public.decisions
        WHERE id=$1
        `,
        [decisionId]
      );
      if (!d.length) return null;

      const { rows: attachments } = await db.query(
        `
        SELECT
          a.id, a.original_name, a.mime_type, a.size_bytes, a.uploaded_at, a.uploaded_by
        FROM public.decision_attachments da
        JOIN public.attachments a ON a.id = da.attachment_id
        WHERE da.decision_id = $1
        ORDER BY a.uploaded_at DESC
        `,
        [decisionId]
      );

      return {
        decision: d[0],
        attachments: attachments.map((r) => ({
          ...r,
          downloadUrl: `/api/app/attachments/${r.id}/download`,
        })),
      };
    });

    if (!data) return res.status(404).json({ error: "Not found" });
    res.json(data);
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "getDecisionDetail", message: e.message });
  }
}

async function getDecisionCaseAndAuthor(db: Db, decisionId: string) {
  const r = await db.query(
    `SELECT id, case_id, created_by, archived_at
     FROM public.decisions
     WHERE id=$1`,
    [decisionId]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Decision not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0] as {
    id: string;
    case_id: string;
    created_by: string | null;
    archived_at: string | null;
  };
}

const PatchDecisionBody = z.object({
  title: z.string().min(2).optional(),
  decisionText: z.string().min(2).optional(),
  status: z
    .enum(["draft", "approved", "rejected", "deferred", "cancelled"])
    .optional(),

  // ✅ nullable + optional: küldheted null-lal is (törlés)
  dueAt: z.string().datetime().nullable().optional(),

  // legacy
  ownerUserId: z.string().uuid().nullable().optional(),
  // ✅ human readable: email | username | uuid | null (törlés)
  ownerUser: z.string().nullable().optional(),
});

export async function patchDecision(req: Request, res: Response) {
  try {
    const decisionRef = z.string().min(1).parse(req.params.decisionRef);
    const b = PatchDecisionBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const decisionId = await resolveDecisionId(db, decisionRef);
      const d = await getDecisionCaseAndAuthor(db, decisionId);

      // author OR editor
      if (!(d.created_by && d.created_by === req.user!.id)) {
        await requireCaseRole(db, d.case_id, "editor");
      }

      const sets: string[] = [];
      const params: any[] = [decisionId];

      const push = (sql: string, val: any) => {
        params.push(val);
        sets.push(sql.replace("$X", `$${params.length}`));
      };

      // title
      if (b.title !== undefined) push(`title=$X`, b.title.trim());

      // decisionText
      if (b.decisionText !== undefined)
        push(`decision_text=$X`, b.decisionText);

      // status
      if (b.status !== undefined)
        push(`status=$X::public.decision_status`, b.status);

      // ✅ dueAt: undefined -> ne nyúlj hozzá; null -> töröld; string -> set
      if (b.dueAt !== undefined) {
        push(`due_at=$X`, b.dueAt ? new Date(b.dueAt) : null);
      }

      // ✅ owner: 3 féle mód
      // 1) ownerUserId (legacy): undefined -> ignore; null -> törlés; uuid -> set
      if (b.ownerUserId !== undefined) {
        push(`owner_user_id=$X`, b.ownerUserId);
      }
      // 2) ownerUser (human): undefined -> ignore; null -> törlés; string -> resolve uuid
      else if (b.ownerUser !== undefined) {
        if (b.ownerUser === null) {
          push(`owner_user_id=$X`, null);
        } else {
          const raw = b.ownerUser.trim();
          const asUuid = z.string().uuid().safeParse(raw);

          if (asUuid.success) push(`owner_user_id=$X`, asUuid.data);
          else if (raw.includes("@"))
            push(`owner_user_id=$X`, await resolveUserId(db, { email: raw }));
          else
            push(
              `owner_user_id=$X`,
              await resolveUserId(db, { username: raw })
            );
        }
      }

      if (!sets.length) return;

      params.push(req.user!.id);

      await db.query(
        `
        UPDATE public.decisions
        SET ${sets.join(", ")},
            updated_by = $${params.length},
            updated_at = now()
        WHERE id=$1 AND archived_at IS NULL
        `,
        params
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "DECISION_UPDATED",
      objectType: "decision",
      objectId: decisionRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { decision_ref: decisionRef, changed: Object.keys(b) },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "patchDecision", message: e.message });
  }
}

export async function archiveDecision(req: Request, res: Response) {
  try {
    const decisionRef = z.string().min(1).parse(req.params.decisionRef);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const decisionId = await resolveDecisionId(db, decisionRef);

      const d = await getDecisionCaseAndAuthor(db, decisionId);

      if (!(d.created_by && d.created_by === req.user!.id)) {
        await requireCaseRole(db, d.case_id, "editor");
      }

      await db.query(
        `
        UPDATE public.decisions
        SET archived_at=now(), archived_by=$2, updated_by=$2, updated_at=now()
        WHERE id=$1 AND archived_at IS NULL
        `,
        [decisionId, req.user!.id]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "DECISION_ARCHIVED",
      objectType: "decision",
      objectId: decisionRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { decision_ref: decisionRef },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "archiveDecision", message: e.message });
  }
}
export async function unarchiveDecision(req: Request, res: Response) {
  try {
    const decisionRef = z.string().min(1).parse(req.params.decisionRef);

    await withRlsTx(rlsCtx(req), async (db) => {
      const decisionId = await resolveDecisionId(db, decisionRef);

      // mirror archive permission: must be editor on decision.case_id
      const { rows } = await db.query<{ case_id: string }>(
        `SELECT case_id FROM public.decisions WHERE id=$1 LIMIT 1`,
        [decisionId]
      );
      if (!rows.length) {
        const e: any = new Error("Decision not found");
        e.statusCode = 404;
        throw e;
      }
      await requireCaseRole(db, rows[0].case_id, "editor");

      await db.query(
        `
        UPDATE public.decisions
        SET archived_at=NULL, archived_by=NULL, updated_by=$2, updated_at=now()
        WHERE id=$1 AND archived_at IS NOT NULL
        `,
        [decisionId, req.user!.id]
      );
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "unarchiveDecision", message: e.message });
  }
}
