// src/app/controllers/api/admin/action-items/action-items.controller.ts
import type { Request, Response } from "express";
import { z } from "zod";
import { withRlsTx, type Db } from "@/lib/postgres.js";
import { requireCaseRole } from "../../../../utils/aclCase.js";
import { writeAuditLog } from "../../../../audit/writeAuditLog.js";
const UUID_RE =
  /^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;
const AssigneeRole = z.enum(["owner", "contributor", "reviewer", "watcher"]);

const PatchAssigneesBody = z.object({
  assignees: z
    .array(
      z.object({
        userId: z.string().uuid(),
        role: AssigneeRole.optional(),
      })
    )
    .default([]),
});

async function resolveCaseId(db: any, caseRef: string): Promise<string> {
  if (UUID_RE.test(caseRef)) return z.string().uuid().parse(caseRef);
  const r = await db.query(
    `SELECT id FROM public.cases WHERE code=$1 LIMIT 1`,
    [caseRef.trim()]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Case not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0].id as string;
}

async function resolveActionItemId(db: any, ref: string): Promise<string> {
  if (UUID_RE.test(ref)) return z.string().uuid().parse(ref);
  const r = await db.query(
    `SELECT id FROM public.action_items WHERE code=$1 LIMIT 1`,
    [ref.trim()]
  );
  if (!r.rows?.length) {
    const e: any = new Error("Action item not found");
    e.statusCode = 404;
    throw e;
  }
  return r.rows[0].id as string;
}
function rlsCtx(req: Request) {
  return {
    userId: req.user!.id,
    hasSensitiveAccess: req.user?.accessLevel?.name === "super-admin",
  };
}

const ListActionItemsQuery = z.object({
  mine: z.union([z.literal("1"), z.literal("true")]).optional(),
  includeArchived: z.union([z.literal("1"), z.literal("true")]).optional(),

  status: z
    .enum(["open", "in_progress", "blocked", "done", "cancelled"])
    .optional(),
  dueBefore: z.string().datetime().optional(),
  caseId: z.string().uuid().optional(),
  q: z.string().optional(),
});

export async function listActionItems(req: Request, res: Response) {
  try {
    const q = ListActionItemsQuery.parse(req.query);

    const items = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const params: any[] = [];
      const where: string[] = [];

      if (q.mine) {
        where.push(`ai.assigned_to = public.app_user_id()`);
      }
      if (!q.includeArchived) {
        where.push(`ai.archived_at IS NULL`);
      }
      if (q.status) {
        params.push(q.status);
        where.push(`ai.status = $${params.length}::public.action_item_status`);
      }
      if (q.dueBefore) {
        params.push(new Date(q.dueBefore));
        where.push(`ai.due_at < $${params.length}`);
      }
      if (q.caseId) {
        params.push(q.caseId);
        where.push(`ai.case_id = $${params.length}::uuid`);
      }
      if (q.q?.trim()) {
        params.push(`%${q.q.trim()}%`);
        where.push(
          `(ai.code ILIKE $${params.length} OR ai.title ILIKE $${params.length} OR ai.description ILIKE $${params.length})`
        );
      }

      const sql = `
        SELECT
          ai.id, ai.code, ai.title, ai.description, ai.status,
          ai.case_id, ai.thread_id, ai.decision_id, ai.meeting_id,
          ai.assigned_to, ai.created_by,
          ai.due_at, ai.created_at, ai.updated_at,
          ai.archived_at, ai.archived_by,
          c.title AS case_title,
          c.code  AS case_code
        FROM public.action_items ai
        LEFT JOIN public.cases c ON c.id = ai.case_id
        ${where.length ? `WHERE ${where.join(" AND ")}` : ""}
        ORDER BY
          (ai.due_at IS NULL) ASC,
          ai.due_at ASC,
          ai.updated_at DESC
        LIMIT 200
      `;

      const { rows } = await db.query(sql, params);
      return rows;
    });

    res.json({ items });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listActionItems", message: e.message });
  }
}

const CreateActionItemBody = z.object({
  title: z.string().min(2),
  description: z.string().optional(),
  status: z
    .enum(["open", "in_progress", "blocked", "done", "cancelled"])
    .optional(),
  caseId: z.string().uuid(),
  threadId: z.string().uuid().optional(),
  decisionId: z.string().uuid().optional(),
  meetingId: z.string().uuid().optional(),
  assignedTo: z.string().uuid().optional(),
  dueAt: z.string().datetime().optional(),
});

export async function createActionItem(req: Request, res: Response) {
  try {
    const b = CreateActionItemBody.parse(req.body);

    const created = await withRlsTx(rlsCtx(req), async (db: Db) => {
      await requireCaseRole(db, b.caseId, "editor");

      const { rows } = await db.query<{ id: string; code: string }>(
        `
        INSERT INTO public.action_items (
          id, title, description, status,
          case_id, thread_id, decision_id, meeting_id,
          assigned_to, created_by, due_at
        )
        VALUES (
          gen_random_uuid(), $1, $2, $3::public.action_item_status,
          $4, $5, $6, $7,
          $8, $9, $10
        )
        RETURNING id, code
        `,
        [
          b.title.trim(),
          b.description ?? null,
          b.status ?? "open",
          b.caseId,
          b.threadId ?? null,
          b.decisionId ?? null,
          b.meetingId ?? null,
          b.assignedTo ?? null,
          req.user!.id,
          b.dueAt ? new Date(b.dueAt) : null,
        ]
      );

      return rows[0];
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "ACTION_ITEM_CREATED",
      objectType: "action_item",
      objectId: created.id,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: {
        code: created.code,
        case_id: b.caseId,
        thread_id: b.threadId,
        decision_id: b.decisionId,
        meeting_id: b.meetingId,
        assigned_to: b.assignedTo,
        due_at: b.dueAt ?? null,
      },
    });

    res.status(201).json({ id: created.id, code: created.code });
  } catch (e: any) {
    res.status(400).json({ error: "createActionItem", message: e.message });
  }
}

const PatchActionItemBody = z.object({
  title: z.string().min(2).optional(),
  // ⚠️ nullable, hogy lehessen törölni
  description: z.string().nullable().optional(),
  status: z
    .enum(["open", "in_progress", "blocked", "done", "cancelled"])
    .optional(),
  // nullable: unassign támogatás
  assignedTo: z.string().uuid().nullable().optional(),
  // nullable: dueAt törlés támogatás
  dueAt: z.string().datetime().nullable().optional(),
});

export async function patchActionItem(req: Request, res: Response) {
  try {
    const actionItemRef = z.string().min(1).parse(req.params.actionItemRef);
    const b = PatchActionItemBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const id = await resolveActionItemId(db, actionItemRef);

      const r = await db.query(
        `SELECT case_id, created_by FROM public.action_items WHERE id=$1`,
        [id]
      );
      const cur = (r.rows ?? []) as Array<{
        case_id: string | null;
        created_by: string | null;
      }>;

      if (!cur.length) {
        const e: any = new Error("Action item not found");
        e.statusCode = 404;
        throw e;
      }

      const caseId = cur[0].case_id;
      const createdBy = cur[0].created_by;

      if (!caseId) {
        const e: any = new Error(
          "Action item without case_id is not supported in v1"
        );
        e.statusCode = 400;
        throw e;
      }

      if (!(createdBy && createdBy === req.user!.id)) {
        await requireCaseRole(db, caseId, "editor");
      }

      const hasTitle = b.title !== undefined;
      const hasDescription = b.description !== undefined;
      const hasStatus = b.status !== undefined;
      const hasAssignedTo = b.assignedTo !== undefined;
      const hasDueAt = b.dueAt !== undefined;

      await db.query(
        `
  UPDATE public.action_items
  SET
    title = CASE
      WHEN $2::boolean THEN $3
      ELSE title
    END,
    description = CASE
      WHEN $4::boolean THEN $5
      ELSE description
    END,
    status = CASE
      WHEN $6::boolean THEN $7::public.action_item_status
      ELSE status
    END,
    assigned_to = CASE
      WHEN $8::boolean THEN $9::uuid
      ELSE assigned_to
    END,
    due_at = CASE
      WHEN $10::boolean THEN $11
      ELSE due_at
    END,
    updated_by = $12,
    updated_at = now()
  WHERE id = $1
  `,
        [
          id,

          // title
          hasTitle,
          hasTitle ? b.title!.trim() : null,

          // description (nullable!)
          hasDescription,
          hasDescription ? b.description ?? null : null,

          // status
          hasStatus,
          hasStatus ? b.status! : null,

          // assigned_to (nullable!)
          hasAssignedTo,
          hasAssignedTo ? b.assignedTo ?? null : null,

          // due_at (nullable!)
          hasDueAt,
          hasDueAt ? (b.dueAt ? new Date(b.dueAt) : null) : null,

          req.user!.id,
        ]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "ACTION_ITEM_UPDATED",
      objectType: "action_item",
      objectId: actionItemRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { action_item_ref: actionItemRef, changed: Object.keys(b) },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "patchActionItem", message: e.message });
  }
}

export async function listActionItemsByCase(req: Request, res: Response) {
  try {
    const caseRef = z.string().min(1).parse(req.params.caseRef);
    const includeArchived =
      req.query.includeArchived === "1" || req.query.includeArchived === "true";

    const items = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const caseId = await resolveCaseId(db, caseRef);

      const { rows } = await db.query(
        `
        SELECT
          ai.id, ai.code, ai.title, ai.description, ai.status,
          ai.thread_id, ai.decision_id, ai.meeting_id,
          ai.assigned_to, ai.created_by,
          ai.due_at, ai.created_at, ai.updated_at,
          ai.archived_at, ai.archived_by
        FROM public.action_items ai
        WHERE ai.case_id = $1
          AND ($2::boolean = true OR ai.archived_at IS NULL)
        ORDER BY
          (ai.due_at IS NULL) ASC,
          ai.due_at ASC,
          ai.updated_at DESC
        `,
        [caseId, includeArchived]
      );
      return rows;
    });

    res.json({ items });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "listActionItemsByCase", message: e.message });
  }
}

export async function archiveActionItem(req: Request, res: Response) {
  try {
    const actionItemRef = z.string().min(1).parse(req.params.actionItemRef);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const id = await resolveActionItemId(db, actionItemRef);

      const r = await db.query(
        `SELECT case_id, created_by FROM public.action_items WHERE id=$1`,
        [id]
      );
      if (!r.rows?.length) {
        const e: any = new Error("Action item not found");
        e.statusCode = 404;
        throw e;
      }
      const caseId = r.rows[0].case_id as string | null;
      const createdBy = r.rows[0].created_by as string | null;

      if (!(createdBy && createdBy === req.user!.id)) {
        if (!caseId) {
          const e: any = new Error(
            "Action item without case_id is not supported in v1"
          );
          e.statusCode = 400;
          throw e;
        }
        await requireCaseRole(db, caseId, "editor");
      }

      await db.query(
        `
        UPDATE public.action_items
        SET archived_at=now(), archived_by=$2, updated_by=$2, updated_at=now()
        WHERE id=$1 AND archived_at IS NULL
        `,
        [id, req.user!.id]
      );
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "ACTION_ITEM_ARCHIVED",
      objectType: "action_item",
      objectId: actionItemRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: { action_item_ref: actionItemRef },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "archiveActionItem", message: e.message });
  }
}
export async function getActionItemAssignees(req: Request, res: Response) {
  try {
    const actionItemRef = z.string().min(1).parse(req.params.actionItemRef);

    const assignees = await withRlsTx(rlsCtx(req), async (db: Db) => {
      const id = await resolveActionItemId(db, actionItemRef);

      const { rows } = await db.query(
        `
        SELECT
          a.user_id,
          a.role,
          u.name,
          u.email,
          u.username
        FROM public.action_item_assignees a
        JOIN public.users u ON u.id = a.user_id
        WHERE a.action_item_id = $1
        ORDER BY
          (a.role = 'owner') DESC,
          u.name ASC
        `,
        [id]
      );

      return rows;
    });

    res.json({ assignees });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "getActionItemAssignees", message: e.message });
  }
}

export async function patchActionItemAssignees(req: Request, res: Response) {
  try {
    const actionItemRef = z.string().min(1).parse(req.params.actionItemRef);
    const b = PatchAssigneesBody.parse(req.body);

    await withRlsTx(rlsCtx(req), async (db: Db) => {
      const id = await resolveActionItemId(db, actionItemRef);

      const r = await db.query(
        `SELECT case_id FROM public.action_items WHERE id=$1`,
        [id]
      );
      if (!r.rows?.length) {
        const e: any = new Error("Action item not found");
        e.statusCode = 404;
        throw e;
      }
      const caseId = r.rows[0].case_id as string | null;
      if (!caseId) {
        const e: any = new Error(
          "Action item without case_id is not supported in v1"
        );
        e.statusCode = 400;
        throw e;
      }

      await requireCaseRole(db, caseId, "editor");

      // replace all assignees atomically
      await db.query(
        `DELETE FROM public.action_item_assignees WHERE action_item_id=$1`,
        [id]
      );

      for (const a of b.assignees) {
        await db.query(
          `
          INSERT INTO public.action_item_assignees(action_item_id, user_id, role, created_by)
          VALUES ($1, $2, $3::public.action_item_assignee_role, $4)
          ON CONFLICT (action_item_id, user_id) DO UPDATE
          SET role = EXCLUDED.role
          `,
          [id, a.userId, a.role ?? "contributor", req.user!.id]
        );
      }

      // Backward-compat: ha van owner, írd be primary owner-nek is
      const owner = b.assignees.find(
        (x) => (x.role ?? "contributor") === "owner"
      );
      if (owner) {
        await db.query(
          `UPDATE public.action_items SET assigned_to=$2, updated_by=$3, updated_at=now() WHERE id=$1`,
          [id, owner.userId, req.user!.id]
        );
      }
    });

    void writeAuditLog({
      actorId: req.user?.id ?? null,
      action: "ACTION_ITEM_ASSIGNEES_UPDATED",
      objectType: "action_item",
      objectId: actionItemRef,
      route: req.originalUrl,
      ip: req.ip,
      payloadHash: null,
      meta: {
        action_item_ref: actionItemRef,
        assignees_count: b.assignees.length,
      },
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "patchActionItemAssignees", message: e.message });
  }
}
export async function unarchiveActionItem(req: Request, res: Response) {
  try {
    const actionItemRef = z.string().min(1).parse(req.params.actionItemRef);

    await withRlsTx(rlsCtx(req), async (db) => {
      const actionItemId = await resolveActionItemId(db, actionItemRef);

      // mirror archive permission: editor on case
      const { rows } = await db.query<{ case_id: string }>(
        `SELECT case_id FROM public.action_items WHERE id=$1 LIMIT 1`,
        [actionItemId]
      );
      if (!rows.length) {
        const e: any = new Error("Action item not found");
        e.statusCode = 404;
        throw e;
      }
      await requireCaseRole(db, rows[0].case_id, "editor");

      await db.query(
        `
        UPDATE public.action_items
        SET archived_at=NULL, archived_by=NULL, updated_by=$2, updated_at=now()
        WHERE id=$1 AND archived_at IS NOT NULL
        `,
        [actionItemId, req.user!.id]
      );
    });

    res.json({ ok: true });
  } catch (e: any) {
    res
      .status(e.statusCode ?? 400)
      .json({ error: "unarchiveActionItem", message: e.message });
  }
}
